import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { MapPin, Clock, DollarSign, Phone, Star, X } from 'lucide-react';
import { rideService } from '../../services/api';
import { useAuth } from '../../hooks/useAuth.jsx';
import Button from '../../components/shared/Button';
import LoadingSpinner from '../../components/shared/LoadingSpinner';
import GoogleMap from './GoogleMap';

const RideScreen = () => {
  const { rideId } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  
  const [ride, setRide] = useState(null);
  const [loading, setLoading] = useState(true);
  const [cancelling, setCancelling] = useState(false);
  const [rating, setRating] = useState(0);
  const [comment, setComment] = useState('');
  const [showRating, setShowRating] = useState(false);

  useEffect(() => {
    loadRide();
    const interval = setInterval(loadRide, 5000); // Atualizar a cada 5 segundos
    
    return () => clearInterval(interval);
  }, [rideId]);

  const loadRide = async () => {
    try {
      const response = await rideService.getRide(rideId);
      if (response.data.success) {
        setRide(response.data.corrida);
        
        // Se a corrida foi concluída, mostrar avaliação
        if (response.data.corrida.status === 'concluida' && !response.data.corrida.avaliacao_motorista) {
          setShowRating(true);
        }
      }
    } catch (error) {
      console.error('Erro ao carregar corrida:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleCancelRide = async () => {
    if (!confirm('Deseja realmente cancelar a corrida?')) return;
    
    setCancelling(true);
    try {
      const response = await rideService.cancelRide(rideId, user.id);
      if (response.data.success) {
        navigate('/');
      }
    } catch (error) {
      console.error('Erro ao cancelar corrida:', error);
    } finally {
      setCancelling(false);
    }
  };

  const handleRateRide = async () => {
    if (rating === 0) return;
    
    try {
      await rideService.rateRide(rideId, rating, comment);
      setShowRating(false);
      navigate('/');
    } catch (error) {
      console.error('Erro ao avaliar corrida:', error);
    }
  };

  const getStatusText = (status) => {
    const statusMap = {
      'solicitada': 'Procurando motorista...',
      'aceita': 'Motorista a caminho',
      'em_andamento': 'Em viagem',
      'concluida': 'Corrida concluída',
      'cancelada': 'Corrida cancelada'
    };
    return statusMap[status] || status;
  };

  const getStatusColor = (status) => {
    const colorMap = {
      'solicitada': 'bg-yellow-500',
      'aceita': 'bg-blue-500',
      'em_andamento': 'bg-green-500',
      'concluida': 'bg-gray-500',
      'cancelada': 'bg-red-500'
    };
    return colorMap[status] || 'bg-gray-500';
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <LoadingSpinner size="lg" text="Carregando corrida..." />
      </div>
    );
  }

  if (!ride) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-600 mb-4">Corrida não encontrada</p>
          <Button onClick={() => navigate('/')}>Voltar</Button>
        </div>
      </div>
    );
  }

  return (
    <div className="relative h-screen">
      {/* Mapa */}
      <GoogleMap
        center={{ 
          latitude: parseFloat(ride.origem_latitude), 
          longitude: parseFloat(ride.origem_longitude) 
        }}
        originCoords={{
          latitude: parseFloat(ride.origem_latitude),
          longitude: parseFloat(ride.origem_longitude)
        }}
        destinationCoords={{
          latitude: parseFloat(ride.destino_latitude),
          longitude: parseFloat(ride.destino_longitude)
        }}
      />

      {/* Status Bar */}
      <div className="absolute top-0 left-0 right-0 z-10">
        <div className={`${getStatusColor(ride.status)} text-white p-4 text-center font-semibold`}>
          {getStatusText(ride.status)}
        </div>
      </div>

      {/* Panel de Informações */}
      <div className="absolute bottom-0 left-0 right-0 bg-white rounded-t-3xl shadow-2xl p-6">
        <div className="w-12 h-1 bg-gray-300 rounded-full mx-auto mb-6"></div>
        
        <h3 className="text-xl font-semibold text-gray-900 mb-4">Detalhes da Corrida</h3>
        
        {/* Endereços */}
        <div className="space-y-3 mb-6">
          <div className="flex items-center space-x-3">
            <div className="w-3 h-3 bg-green-500 rounded-full"></div>
            <span className="text-gray-700">{ride.origem_endereco}</span>
          </div>
          <div className="flex items-center space-x-3">
            <div className="w-3 h-3 bg-red-500 rounded-full"></div>
            <span className="text-gray-700">{ride.destino_endereco}</span>
          </div>
        </div>

        {/* Informações do Motorista */}
        {ride.motorista_nome && (
          <div className="bg-gray-50 rounded-lg p-4 mb-6">
            <h4 className="font-semibold text-gray-900 mb-2">Seu Motorista</h4>
            <div className="flex items-center justify-between">
              <div>
                <p className="font-medium">{ride.motorista_nome}</p>
                <p className="text-sm text-gray-600">Categoria: {ride.categoria_nome}</p>
              </div>
              <Button variant="outline" size="sm">
                <Phone size={16} />
              </Button>
            </div>
          </div>
        )}

        {/* Informações de Preço */}
        <div className="flex items-center justify-between mb-6">
          <span className="text-gray-600">Valor da corrida:</span>
          <span className="text-xl font-bold text-green-600">
            R$ {parseFloat(ride.valor_total).toFixed(2)}
          </span>
        </div>

        {/* Botões de Ação */}
        {ride.status === 'solicitada' || ride.status === 'aceita' ? (
          <Button
            onClick={handleCancelRide}
            variant="danger"
            size="lg"
            loading={cancelling}
            className="w-full"
            icon={<X size={20} />}
          >
            Cancelar Corrida
          </Button>
        ) : ride.status === 'concluida' ? (
          <Button
            onClick={() => navigate('/')}
            variant="primary"
            size="lg"
            className="w-full"
          >
            Nova Corrida
          </Button>
        ) : null}
      </div>

      {/* Modal de Avaliação */}
      {showRating && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-2xl p-6 w-full max-w-md">
            <h3 className="text-xl font-semibold text-center mb-6">Avalie sua corrida</h3>
            
            {/* Estrelas */}
            <div className="flex justify-center space-x-2 mb-6">
              {[1, 2, 3, 4, 5].map((star) => (
                <button
                  key={star}
                  onClick={() => setRating(star)}
                  className={`text-3xl ${
                    star <= rating ? 'text-yellow-400' : 'text-gray-300'
                  }`}
                >
                  <Star fill={star <= rating ? 'currentColor' : 'none'} />
                </button>
              ))}
            </div>

            {/* Comentário */}
            <textarea
              value={comment}
              onChange={(e) => setComment(e.target.value)}
              placeholder="Comentário (opcional)"
              className="w-full p-3 border border-gray-300 rounded-lg mb-6 resize-none"
              rows={3}
            />

            {/* Botões */}
            <div className="flex space-x-3">
              <Button
                onClick={() => setShowRating(false)}
                variant="outline"
                className="flex-1"
              >
                Pular
              </Button>
              <Button
                onClick={handleRateRide}
                variant="primary"
                className="flex-1"
                disabled={rating === 0}
              >
                Avaliar
              </Button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default RideScreen;