import React, { useState, useEffect } from 'react';
import { 
  MapPin, 
  Navigation, 
  Car, 
  Clock, 
  DollarSign, 
  User, 
  LogOut,
  Menu,
  Search,
  Star,
  ChevronRight,
  Home,
  Briefcase,
  Heart
} from 'lucide-react';
import { useGeolocation } from '../../hooks/useGeolocation';
import { adminService, rideService } from '../../services/api';
import VehicleSelectionModal from './VehicleSelectionModal';
import GoogleMap from './GoogleMap';

const MainScreen = ({ user, onLogout }) => {
  const { location, error: locationError } = useGeolocation();
  
  const [origin, setOrigin] = useState('');
  const [destination, setDestination] = useState('');
  const [originCoords, setOriginCoords] = useState(null);
  const [destinationCoords, setDestinationCoords] = useState(null);
  const [categories, setCategories] = useState([]);
  const [showVehicleModal, setShowVehicleModal] = useState(false);
  const [showSideMenu, setShowSideMenu] = useState(false);
  const [loading, setLoading] = useState(false);
  const [notification, setNotification] = useState(null);
  const [currentRide, setCurrentRide] = useState(null);
  const [searchFocused, setSearchFocused] = useState(false);

  useEffect(() => {
    loadCategories();
  }, []);

  useEffect(() => {
    if (location) {
      // Geocoding reverso para preencher origem automaticamente
      reverseGeocode(location.latitude, location.longitude)
        .then(address => {
          setOrigin(address);
          setOriginCoords(location);
        })
        .catch(console.error);
    }
  }, [location]);

  const loadCategories = async () => {
    try {
      const response = await adminService.getCategories();
      if (response.data.success) {
        setCategories(response.data.categorias.filter(c => c.ativo));
      }
    } catch (error) {
      console.error('Erro ao carregar categorias:', error);
      showNotification('Erro ao carregar categorias de veículos', 'error');
    }
  };

  const reverseGeocode = async (lat, lng) => {
    return new Promise((resolve, reject) => {
      const geocoder = new window.google.maps.Geocoder();
      geocoder.geocode(
        { location: { lat, lng } },
        (results, status) => {
          if (status === 'OK' && results[0]) {
            resolve(results[0].formatted_address);
          } else {
            reject('Endereço não encontrado');
          }
        }
      );
    });
  };

  const handleAddressSelect = (address, coords, type) => {
    if (type === 'origin') {
      setOrigin(address);
      setOriginCoords(coords);
    } else {
      setDestination(address);
      setDestinationCoords(coords);
    }
  };

  const handleShowVehicles = () => {
    if (!origin || !destination) {
      showNotification('Preencha origem e destino', 'warning');
      return;
    }

    if (!originCoords || !destinationCoords) {
      showNotification('Aguarde o cálculo das coordenadas', 'warning');
      return;
    }

    setShowVehicleModal(true);
  };

  const handleRideRequest = async (categoryId, estimatedPrice) => {
    setLoading(true);
    
    try {
      const rideData = {
        usuario_id: user.id,
        categoria_veiculo_id: categoryId,
        tipo: 'passageiro',
        origem_latitude: originCoords.latitude,
        origem_longitude: originCoords.longitude,
        origem_endereco: origin,
        destino_latitude: destinationCoords.latitude,
        destino_longitude: destinationCoords.longitude,
        destino_endereco: destination,
        metodo_pagamento: 'dinheiro',
      };

      const response = await rideService.requestRide(rideData);
      
      if (response.data.success) {
        showNotification('Corrida solicitada com sucesso!', 'success');
        setShowVehicleModal(false);
        // Aqui você pode navegar para a tela de acompanhamento da corrida
        // navigate(`/ride/${response.data.corrida_id}`);
      } else {
        showNotification(response.data.message, 'error');
      }
    } catch (error) {
      console.error('Erro ao solicitar corrida:', error);
      showNotification('Erro ao solicitar corrida', 'error');
    } finally {
      setLoading(false);
    }
  };

  const showNotification = (message, type = 'info') => {
    setNotification({ message, type });
    setTimeout(() => setNotification(null), 5000);
  };

  return (
    <div className="relative h-screen bg-gray-100">
      {/* Google Maps */}
      <GoogleMap
        center={location || { latitude: -23.550520, longitude: -46.633308 }}
        userLocation={location}
        origin={originCoords}
        destination={destinationCoords}
      />

      {/* Header Limpo */}
      <div className="absolute top-0 left-0 right-0 z-20">
        <div className="flex items-center justify-between p-4">
          <button
            onClick={() => setShowSideMenu(true)}
            className="w-12 h-12 bg-white rounded-2xl shadow-lg flex items-center justify-center"
          >
            <Menu size={20} className="text-gray-700" />
          </button>
          
          <div className="flex-1 mx-4">
            <div className="bg-white rounded-2xl shadow-lg px-4 py-3">
              <div className="flex items-center gap-3">
                <Search size={18} className="text-gray-400" />
                <input
                  type="text"
                  placeholder="Para onde?"
                  value={destination}
                  onChange={(e) => setDestination(e.target.value)}
                  onFocus={() => setSearchFocused(true)}
                  className="flex-1 text-gray-900 placeholder-gray-400 bg-transparent outline-none font-medium"
                />
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Painel de Busca Expandido */}
      {searchFocused && (
        <div className="absolute top-0 left-0 right-0 bottom-0 z-30 bg-white">
          <div className="p-4">
            {/* Header do painel de busca */}
            <div className="flex items-center gap-3 mb-6">
              <button
                onClick={() => setSearchFocused(false)}
                className="w-10 h-10 rounded-full flex items-center justify-center hover:bg-gray-100"
              >
                <ChevronRight size={20} className="text-gray-600 rotate-180" />
              </button>
              <h2 className="text-lg font-semibold text-gray-900">Para onde?</h2>
            </div>

            {/* Campos de endereço */}
            <div className="space-y-4 mb-6">
              <div className="flex items-center gap-4 p-4 bg-gray-50 rounded-2xl">
                <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
                <input
                  type="text"
                  placeholder="Origem"
                  value={origin}
                  onChange={(e) => setOrigin(e.target.value)}
                  className="flex-1 bg-transparent outline-none text-gray-900 placeholder-gray-500"
                />
              </div>
              
              <div className="flex items-center gap-4 p-4 bg-gray-50 rounded-2xl">
                <div className="w-3 h-3 bg-red-500 rounded-full"></div>
                <input
                  type="text"
                  placeholder="Destino"
                  value={destination}
                  onChange={(e) => setDestination(e.target.value)}
                  className="flex-1 bg-transparent outline-none text-gray-900 placeholder-gray-500"
                />
              </div>
            </div>

            {/* Locais salvos */}
            <div className="space-y-3">
              <h3 className="text-sm font-medium text-gray-500 uppercase tracking-wide">Locais Salvos</h3>
              
              <div className="space-y-2">
                <button className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors">
                  <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                    <Home size={18} className="text-blue-600" />
                  </div>
                  <div className="flex-1 text-left">
                    <p className="font-medium text-gray-900">Casa</p>
                    <p className="text-sm text-gray-500">Adicionar endereço residencial</p>
                  </div>
                </button>

                <button className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors">
                  <div className="w-10 h-10 bg-orange-100 rounded-full flex items-center justify-center">
                    <Briefcase size={18} className="text-orange-600" />
                  </div>
                  <div className="flex-1 text-left">
                    <p className="font-medium text-gray-900">Trabalho</p>
                    <p className="text-sm text-gray-500">Adicionar endereço comercial</p>
                  </div>
                </button>
              </div>
            </div>

            {/* Botão de confirmar */}
            {origin && destination && (
              <div className="fixed bottom-6 left-4 right-4">
                <button
                  onClick={() => {
                    setSearchFocused(false);
                    handleShowVehicles();
                  }}
                  className="w-full bg-black text-white py-4 rounded-2xl font-semibold text-lg shadow-lg"
                >
                  Confirmar Endereços
                </button>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Botões de Ação Rápida */}
      {!searchFocused && (
        <div className="absolute bottom-6 left-4 right-4 z-20">
          <div className="bg-white rounded-3xl shadow-xl p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Viagens Rápidas</h3>
            
            <div className="grid grid-cols-2 gap-3">
              <button className="flex flex-col items-center p-4 bg-gray-50 rounded-2xl hover:bg-gray-100 transition-colors">
                <div className="w-12 h-12 bg-black rounded-full flex items-center justify-center mb-2">
                  <Car size={20} className="text-white" />
                </div>
                <span className="text-sm font-medium text-gray-900">Viagem</span>
                <span className="text-xs text-gray-500">Rápida e segura</span>
              </button>

              <button className="flex flex-col items-center p-4 bg-gray-50 rounded-2xl hover:bg-gray-100 transition-colors">
                <div className="w-12 h-12 bg-orange-500 rounded-full flex items-center justify-center mb-2">
                  <Navigation size={20} className="text-white" />
                </div>
                <span className="text-sm font-medium text-gray-900">Entrega</span>
                <span className="text-xs text-gray-500">Envie objetos</span>
              </button>
            </div>

            {/* Promoções */}
            <div className="mt-4 p-4 bg-gradient-to-r from-blue-50 to-purple-50 rounded-2xl border border-blue-100">
              <div className="flex items-center gap-3">
                <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center">
                  <Star size={16} className="text-white" />
                </div>
                <div className="flex-1">
                  <p className="text-sm font-medium text-gray-900">Primeira viagem grátis!</p>
                  <p className="text-xs text-gray-600">Use o código BEMVINDO</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Menu Lateral */}
      {showSideMenu && (
        <div className="fixed inset-0 z-50">
          <div 
            className="absolute inset-0 bg-black bg-opacity-50"
            onClick={() => setShowSideMenu(false)}
          />
          <div className="absolute top-0 left-0 w-80 h-full bg-white">
            <div className="p-6 pt-12">
              <div className="flex items-center gap-4 mb-8">
                <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                  <User size={24} className="text-white" />
                </div>
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">{user?.nome || 'Usuário'}</h3>
                  <div className="flex items-center gap-1">
                    <Star size={14} className="text-yellow-500 fill-current" />
                    <span className="text-sm text-gray-600">4.9</span>
                  </div>
                </div>
              </div>
              
              <div className="space-y-2">
                <button className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors">
                  <Clock size={20} className="text-gray-500" />
                  <span className="text-gray-900">Suas viagens</span>
                </button>
                
                <button className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors">
                  <DollarSign size={20} className="text-gray-500" />
                  <span className="text-gray-900">Pagamentos</span>
                </button>
                
                <button className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors">
                  <Heart size={20} className="text-gray-500" />
                  <span className="text-gray-900">Locais salvos</span>
                </button>
                
                <div className="border-t pt-4 mt-4">
                  <button 
                    onClick={onLogout}
                    className="w-full flex items-center gap-4 p-4 hover:bg-gray-50 rounded-2xl transition-colors text-red-600"
                  >
                    <LogOut size={20} />
                    <span>Sair</span>
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal de Seleção de Veículo */}
      <VehicleSelectionModal
        isOpen={showVehicleModal}
        onClose={() => setShowVehicleModal(false)}
        categories={categories}
        origin={origin}
        destination={destination}
        originCoords={originCoords}
        destinationCoords={destinationCoords}
        onRideRequest={handleRideRequest}
        loading={loading}
      />

      {/* Notificação */}
      {notification && (
        <div className="fixed top-20 left-4 right-4 z-50">
          <div className={`p-4 rounded-2xl shadow-lg ${
            notification.type === 'success' ? 'bg-green-500 text-white' :
            notification.type === 'error' ? 'bg-red-500 text-white' :
            'bg-blue-500 text-white'
          }`}>
            {notification.message}
          </div>
        </div>
      )}
    </div>
  );
};

export default MainScreen;