import { useRef, useCallback } from 'react';

export const useSound = () => {
  const audioContextRef = useRef(null);
  const currentAudioRef = useRef(null);

  // Inicializar AudioContext
  const initAudioContext = useCallback(() => {
    if (!audioContextRef.current) {
      audioContextRef.current = new (window.AudioContext || window.webkitAudioContext)();
    }
    return audioContextRef.current;
  }, []);

  // Tocar som de nova corrida (Som do Uber)
  const playNewRideSound = useCallback(() => {
    try {
      console.log('🔊 Tentando tocar som do Uber...');
      
      // Parar som anterior se estiver tocando
      if (currentAudioRef.current) {
        currentAudioRef.current.pause();
        currentAudioRef.current.currentTime = 0;
      }
      
      // Criar elemento de áudio sem CORS
      const audio = new Audio();
      audio.volume = 0.8;
      audio.preload = 'auto';
      audio.loop = true; // Loop para continuar tocando
      
      // Salvar referência do áudio atual
      currentAudioRef.current = audio;
      
      // Configurar eventos
      audio.addEventListener('canplaythrough', () => {
        console.log('✅ Som do Uber carregado, tocando...');
        audio.play().then(() => {
          console.log('🎵 Som do Uber tocado com sucesso!');
        }).catch(error => {
          console.warn('❌ Erro ao reproduzir:', error);
          playFallbackNewRideSound();
        });
      });
      
      audio.addEventListener('error', (error) => {
        console.warn('❌ Erro ao carregar som do Uber:', error);
        playFallbackNewRideSound();
      });
      
      // Definir source por último
      audio.src = 'https://servebot.online/som/Uber.mp3';
      
      // Tentar tocar imediatamente também (para casos onde já está em cache)
      setTimeout(() => {
        audio.play().catch(() => {
          // Se falhar, o fallback já foi configurado nos eventos
        });
      }, 100);
      
      // Vibrar se disponível
      if (navigator.vibrate) {
        navigator.vibrate([200, 100, 200, 100, 200]);
      }
    } catch (error) {
      console.warn('Erro geral ao tocar som do Uber:', error);
      playFallbackNewRideSound();
    }
  }, []);

  // Parar som atual
  const stopCurrentSound = useCallback(() => {
    try {
      if (currentAudioRef.current) {
        currentAudioRef.current.pause();
        currentAudioRef.current.currentTime = 0;
        currentAudioRef.current = null;
        console.log('🔇 Som parado');
      }
    } catch (error) {
      console.warn('Erro ao parar som:', error);
    }
  }, []);

  // Fallback para som sintético de nova corrida (melhorado)
  const playFallbackNewRideSound = useCallback(() => {
    try {
      console.log('🔄 Usando som sintético melhorado como fallback...');
      const audioContext = initAudioContext();
      
      // Som mais elaborado que imita notificação do Uber
      const createTone = (frequency, startTime, duration) => {
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        oscillator.frequency.setValueAtTime(frequency, startTime);
        oscillator.type = 'sine';
        
        gainNode.gain.setValueAtTime(0, startTime);
        gainNode.gain.linearRampToValueAtTime(0.3, startTime + 0.01);
        gainNode.gain.exponentialRampToValueAtTime(0.01, startTime + duration);
        
        oscillator.start(startTime);
        oscillator.stop(startTime + duration);
      };
      
      // Sequência de tons que imita o Uber
      const now = audioContext.currentTime;
      createTone(659.25, now, 0.15);        // E5
      createTone(783.99, now + 0.15, 0.15); // G5
      createTone(987.77, now + 0.3, 0.3);   // B5 (mais longo)
      
      console.log('🎵 Som sintético melhorado tocado!');
    } catch (error) {
      console.warn('❌ Fallback de áudio também falhou:', error);
      // Último recurso: beep simples
      try {
        const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blmGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
        audio.volume = 0.5;
        audio.play().catch(() => console.log('Último recurso também falhou'));
      } catch (e) {
        console.log('Todos os métodos de áudio falharam');
      }
    }
  }, [initAudioContext]);

  // Tocar som de cancelamento
  const playCancelSound = useCallback(() => {
    try {
      const audioContext = initAudioContext();
      
      const oscillator = audioContext.createOscillator();
      const gainNode = audioContext.createGain();
      
      oscillator.connect(gainNode);
      gainNode.connect(audioContext.destination);
      
      // Som mais grave para cancelamento
      oscillator.frequency.setValueAtTime(400, audioContext.currentTime);
      oscillator.frequency.setValueAtTime(300, audioContext.currentTime + 0.1);
      oscillator.frequency.setValueAtTime(200, audioContext.currentTime + 0.2);
      
      gainNode.gain.setValueAtTime(0.2, audioContext.currentTime);
      gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
      
      oscillator.start(audioContext.currentTime);
      oscillator.stop(audioContext.currentTime + 0.3);
      
      // Vibração curta
      if (navigator.vibrate) {
        navigator.vibrate([100]);
      }
    } catch (error) {
      console.warn('Erro ao tocar som de cancelamento:', error);
    }
  }, [initAudioContext]);

  // Tocar som de sucesso
  const playSuccessSound = useCallback(() => {
    try {
      const audioContext = initAudioContext();
      
      const oscillator = audioContext.createOscillator();
      const gainNode = audioContext.createGain();
      
      oscillator.connect(gainNode);
      gainNode.connect(audioContext.destination);
      
      // Som ascendente para sucesso
      oscillator.frequency.setValueAtTime(600, audioContext.currentTime);
      oscillator.frequency.setValueAtTime(800, audioContext.currentTime + 0.1);
      oscillator.frequency.setValueAtTime(1000, audioContext.currentTime + 0.2);
      
      gainNode.gain.setValueAtTime(0.2, audioContext.currentTime);
      gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
      
      oscillator.start(audioContext.currentTime);
      oscillator.stop(audioContext.currentTime + 0.3);
      
      // Vibração de sucesso
      if (navigator.vibrate) {
        navigator.vibrate([50, 50, 50]);
      }
    } catch (error) {
      console.warn('Erro ao tocar som de sucesso:', error);
    }
  }, [initAudioContext]);

  return {
    playNewRideSound,
    playCancelSound,
    playSuccessSound,
    stopCurrentSound
  };
};