import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { 
  MapPin, 
  Navigation, 
  Phone, 
  Play, 
  Square, 
  CheckCircle,
  Clock,
  DollarSign,
  User
} from 'lucide-react';
import { rideService, driverService } from '../../services/api';
import { useAuth } from '../../hooks/useAuth.jsx';
import Button from '../../components/shared/Button';
import LoadingSpinner from '../../components/shared/LoadingSpinner';
import GoogleMap from './GoogleMap';

const RideScreen = () => {
  const { rideId } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  
  const [ride, setRide] = useState(null);
  const [loading, setLoading] = useState(true);
  const [actionLoading, setActionLoading] = useState(false);

  useEffect(() => {
    loadRide();
    const interval = setInterval(loadRide, 5000); // Atualizar a cada 5 segundos
    
    return () => clearInterval(interval);
  }, [rideId]);

  const loadRide = async () => {
    try {
      const response = await rideService.getRide(rideId);
      if (response.data.success) {
        setRide(response.data.corrida);
      }
    } catch (error) {
      console.error('Erro ao carregar corrida:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStartRide = async () => {
    setActionLoading(true);
    try {
      const response = await driverService.startRide(rideId, user.id);
      if (response.data.success) {
        loadRide(); // Recarregar dados
      }
    } catch (error) {
      console.error('Erro ao iniciar corrida:', error);
    } finally {
      setActionLoading(false);
    }
  };

  const handleCompleteRide = async () => {
    if (!confirm('Confirma a conclusão da corrida?')) return;
    
    setActionLoading(true);
    try {
      const response = await driverService.completeRide(rideId, user.id);
      if (response.data.success) {
        navigate('/');
      }
    } catch (error) {
      console.error('Erro ao concluir corrida:', error);
    } finally {
      setActionLoading(false);
    }
  };

  const getStatusText = (status) => {
    const statusMap = {
      'aceita': 'Vá até o passageiro',
      'em_andamento': 'Corrida em andamento',
      'concluida': 'Corrida concluída'
    };
    return statusMap[status] || status;
  };

  const getStatusColor = (status) => {
    const colorMap = {
      'aceita': 'bg-blue-500',
      'em_andamento': 'bg-green-500',
      'concluida': 'bg-gray-500'
    };
    return colorMap[status] || 'bg-gray-500';
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <LoadingSpinner size="lg" text="Carregando corrida..." />
      </div>
    );
  }

  if (!ride) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-600 mb-4">Corrida não encontrada</p>
          <Button onClick={() => navigate('/')}>Voltar</Button>
        </div>
      </div>
    );
  }

  return (
    <div className="relative h-screen">
      {/* Mapa */}
      <GoogleMap
        center={{ 
          latitude: parseFloat(ride.origem_latitude), 
          longitude: parseFloat(ride.origem_longitude) 
        }}
        originCoords={{
          latitude: parseFloat(ride.origem_latitude),
          longitude: parseFloat(ride.origem_longitude)
        }}
        destinationCoords={{
          latitude: parseFloat(ride.destino_latitude),
          longitude: parseFloat(ride.destino_longitude)
        }}
      />

      {/* Status Bar */}
      <div className="absolute top-0 left-0 right-0 z-10">
        <div className={`${getStatusColor(ride.status)} text-white p-4 text-center font-semibold`}>
          {getStatusText(ride.status)}
        </div>
      </div>

      {/* Panel de Informações */}
      <div className="absolute bottom-0 left-0 right-0 bg-white rounded-t-3xl shadow-2xl p-6">
        <div className="w-12 h-1 bg-gray-300 rounded-full mx-auto mb-6"></div>
        
        {/* Informações do Passageiro */}
        <div className="bg-gray-50 rounded-lg p-4 mb-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center">
                <User className="w-6 h-6 text-white" />
              </div>
              <div>
                <h4 className="font-semibold text-gray-900">{ride.usuario_nome}</h4>
                <p className="text-sm text-gray-600">Passageiro</p>
              </div>
            </div>
            
            <Button variant="outline" size="sm">
              <Phone size={16} />
            </Button>
          </div>
        </div>

        {/* Endereços */}
        <div className="space-y-3 mb-6">
          <div className="flex items-center space-x-3">
            <MapPin className="w-5 h-5 text-green-500" />
            <div>
              <p className="text-sm text-gray-600">Origem</p>
              <p className="font-medium text-gray-900">{ride.origem_endereco}</p>
            </div>
          </div>
          
          <div className="flex items-center space-x-3">
            <Navigation className="w-5 h-5 text-red-500" />
            <div>
              <p className="text-sm text-gray-600">Destino</p>
              <p className="font-medium text-gray-900">{ride.destino_endereco}</p>
            </div>
          </div>
        </div>

        {/* Informações da Corrida */}
        <div className="grid grid-cols-3 gap-4 mb-6">
          <div className="text-center">
            <div className="flex items-center justify-center mb-1">
              <Clock className="w-4 h-4 text-gray-500" />
            </div>
            <p className="text-xs text-gray-600">Tempo</p>
            <p className="font-semibold">{ride.duracao_minutos}min</p>
          </div>
          
          <div className="text-center">
            <div className="flex items-center justify-center mb-1">
              <Navigation className="w-4 h-4 text-gray-500" />
            </div>
            <p className="text-xs text-gray-600">Distância</p>
            <p className="font-semibold">{ride.distancia_km}km</p>
          </div>
          
          <div className="text-center">
            <div className="flex items-center justify-center mb-1">
              <DollarSign className="w-4 h-4 text-gray-500" />
            </div>
            <p className="text-xs text-gray-600">Ganho</p>
            <p className="font-semibold text-green-600">
              R$ {parseFloat(ride.valor_motorista).toFixed(2)}
            </p>
          </div>
        </div>

        {/* Botões de Ação */}
        {ride.status === 'aceita' && (
          <Button
            onClick={handleStartRide}
            variant="success"
            size="lg"
            loading={actionLoading}
            className="w-full bg-green-600 hover:bg-green-700"
            icon={<Play size={20} />}
          >
            Iniciar Corrida
          </Button>
        )}

        {ride.status === 'em_andamento' && (
          <Button
            onClick={handleCompleteRide}
            variant="primary"
            size="lg"
            loading={actionLoading}
            className="w-full"
            icon={<CheckCircle size={20} />}
          >
            Finalizar Corrida
          </Button>
        )}

        {ride.status === 'concluida' && (
          <Button
            onClick={() => navigate('/')}
            variant="primary"
            size="lg"
            className="w-full"
          >
            Voltar ao Início
          </Button>
        )}
      </div>
    </div>
  );
};

export default RideScreen;