import React, { useEffect, useRef, useState } from 'react';

const GoogleMap = ({ 
  center, 
  driverLocation, 
  availableRides = [], 
  isOnline = false,
  isDarkMap = false,
  showSurgeHexagons = false 
}) => {
  const mapRef = useRef(null);
  const mapInstanceRef = useRef(null);
  const driverMarkerRef = useRef(null);
  const rideMarkersRef = useRef([]);
  const surgeOverlaysRef = useRef([]);
  const [mapLoaded, setMapLoaded] = useState(false);

  useEffect(() => {
    const checkGoogleMaps = () => {
      if (window.google && window.google.maps && mapRef.current) {
        initializeMap();
        setMapLoaded(true);
      } else {
        setTimeout(checkGoogleMaps, 100);
      }
    };
    
    checkGoogleMaps();
  }, []);

  useEffect(() => {
    if (mapInstanceRef.current && center) {
      mapInstanceRef.current.setCenter({
        lat: center.latitude,
        lng: center.longitude
      });
    }
  }, [center]);

  useEffect(() => {
    updateDriverMarker();
  }, [driverLocation, isOnline, mapLoaded]);

  useEffect(() => {
    if (mapInstanceRef.current) {
      mapInstanceRef.current.setOptions({ styles: getMapStyles() });
    }
  }, [isDarkMap]);

  useEffect(() => {
    if (showSurgeHexagons && isOnline) {
      addSurgeOverlays();
    } else {
      removeSurgeOverlays();
    }
  }, [showSurgeHexagons, isOnline, mapLoaded]);

  const getMapStyles = () => {
    if (isDarkMap) {
      return [
        { elementType: "geometry", stylers: [{ color: "#242f3e" }] },
        { elementType: "labels.text.stroke", stylers: [{ color: "#242f3e" }] },
        { elementType: "labels.text.fill", stylers: [{ color: "#746855" }] },
        {
          featureType: "administrative.locality",
          elementType: "labels.text.fill",
          stylers: [{ color: "#d59563" }]
        },
        {
          featureType: "poi",
          elementType: "labels.text.fill",
          stylers: [{ color: "#d59563" }]
        },
        {
          featureType: "poi.park",
          elementType: "geometry",
          stylers: [{ color: "#263c3f" }]
        },
        {
          featureType: "road",
          elementType: "geometry",
          stylers: [{ color: "#38414e" }]
        },
        {
          featureType: "road",
          elementType: "geometry.stroke",
          stylers: [{ color: "#212a37" }]
        },
        {
          featureType: "water",
          elementType: "geometry",
          stylers: [{ color: "#17263c" }]
        },
        {
          featureType: 'poi',
          elementType: 'labels',
          stylers: [{ visibility: 'off' }]
        }
      ];
    } else {
      return [
        {
          featureType: 'poi',
          elementType: 'labels',
          stylers: [{ visibility: 'off' }]
        }
      ];
    }
  };

  const initializeMap = () => {
    try {
      if (!window.google || !window.google.maps) {
        console.warn('Google Maps não está disponível');
        return;
      }

      const mapOptions = {
        center: {
          lat: center?.latitude || -23.550520,
          lng: center?.longitude || -46.633308
        },
        zoom: 15,
        styles: getMapStyles(),
        disableDefaultUI: true,
        zoomControl: true,
        mapTypeControl: false,
        streetViewControl: false,
        fullscreenControl: false,
      };

      mapInstanceRef.current = new window.google.maps.Map(mapRef.current, mapOptions);
      
      // Obter localização atual
      if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
          (position) => {
            const pos = {
              lat: position.coords.latitude,
              lng: position.coords.longitude
            };
            mapInstanceRef.current.setCenter(pos);
            updateDriverMarker({
              latitude: position.coords.latitude,
              longitude: position.coords.longitude
            });
          },
          () => {
            console.warn('Erro ao obter localização');
          }
        );
      }
    } catch (error) {
      console.error('Erro ao inicializar mapa:', error);
    }
  };

  const updateDriverMarker = (location = driverLocation) => {
    if (!mapInstanceRef.current || !location) return;

    // Remover marcador anterior
    if (driverMarkerRef.current) {
      driverMarkerRef.current.setMap(null);
    }

    const strokeColor = isDarkMap ? '#242f3e' : 'white';
    const carColor = isDarkMap ? '#d59563' : 'white';
    
    // Criar SVG customizado para o marcador
    const svgIcon = `
      <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
        <circle cx="20" cy="20" r="18" fill="#FF6B35" stroke="${strokeColor}" stroke-width="4"/>
        <path d="M12 18h16l-2 8H14l-2-8z" fill="${carColor}"/>
        <circle cx="16" cy="28" r="2" fill="${carColor}"/>
        <circle cx="24" cy="28" r="2" fill="${carColor}"/>
      </svg>
    `;

    // Criar novo marcador do motorista
    driverMarkerRef.current = new window.google.maps.Marker({
      position: {
        lat: location.latitude,
        lng: location.longitude
      },
      map: mapInstanceRef.current,
      icon: {
        url: 'data:image/svg+xml;base64,' + btoa(svgIcon),
        scaledSize: new window.google.maps.Size(40, 40),
        anchor: new window.google.maps.Point(20, 20)
      },
      title: 'Sua localização',
      zIndex: 1000
    });

    // Centralizar mapa na localização do motorista
    mapInstanceRef.current.setCenter({
      lat: location.latitude,
      lng: location.longitude
    });
  };

  // Adicionar hexágonos de surge pricing
  const addSurgeOverlays = () => {
    if (!mapInstanceRef.current) return;

    const surgeAreas = [
      { lat: -23.550520, lng: -46.633308, multiplier: '1,8-2,2x', level: 'high' },
      { lat: -23.545520, lng: -46.628308, multiplier: '1,5-1,9x', level: 'medium' },
      { lat: -23.555520, lng: -46.638308, multiplier: '1,3-1,7x', level: 'low' },
      { lat: -23.560520, lng: -46.643308, multiplier: '1,6-2,0x', level: 'high' },
      { lat: -23.540520, lng: -46.623308, multiplier: '1,1-1,5x', level: 'low' },
      { lat: -23.565520, lng: -46.648308, multiplier: '1,4-1,8x', level: 'medium' }
    ];

    surgeAreas.forEach(area => {
      const overlay = new window.google.maps.OverlayView();
      
      overlay.onAdd = function() {
        const div = document.createElement('div');
        div.style.position = 'absolute';
        div.style.background = area.level === 'high' ? 'rgba(255, 107, 53, 0.8)' :
                              area.level === 'medium' ? 'rgba(255, 152, 0, 0.8)' :
                              'rgba(76, 175, 80, 0.8)';
        div.style.color = 'white';
        div.style.padding = '4px 8px';
        div.style.borderRadius = '12px';
        div.style.fontSize = '12px';
        div.style.fontWeight = '600';
        div.style.pointerEvents = 'none';
        div.innerHTML = `⚡ ${area.multiplier}`;
        
        const panes = this.getPanes();
        panes.overlayLayer.appendChild(div);
        this.div = div;
      };
      
      overlay.draw = function() {
        const projection = this.getProjection();
        const position = projection.fromLatLngToDivPixel(
          new window.google.maps.LatLng(area.lat, area.lng)
        );
        
        if (this.div) {
          this.div.style.left = (position.x - 40) + 'px';
          this.div.style.top = (position.y - 15) + 'px';
        }
      };
      
      overlay.onRemove = function() {
        if (this.div) {
          this.div.parentNode.removeChild(this.div);
          this.div = null;
        }
      };
      
      overlay.setMap(mapInstanceRef.current);
      surgeOverlaysRef.current.push(overlay);
    });
  };

  const removeSurgeOverlays = () => {
    surgeOverlaysRef.current.forEach(overlay => {
      overlay.setMap(null);
    });
    surgeOverlaysRef.current = [];
  };

  // Fallback se o Google Maps não estiver disponível
  if (!window.google && !mapLoaded) {
    return (
      <div 
        className="w-full h-full flex items-center justify-center"
        style={{ 
          minHeight: '400px',
          backgroundColor: '#f3f4f6',
          backgroundImage: 'linear-gradient(45deg, #e5e7eb 25%, transparent 25%), linear-gradient(-45deg, #e5e7eb 25%, transparent 25%), linear-gradient(45deg, transparent 75%, #e5e7eb 75%), linear-gradient(-45deg, transparent 75%, #e5e7eb 75%)',
          backgroundSize: '20px 20px',
          backgroundPosition: '0 0, 0 10px, 10px -10px, -10px 0px'
        }}
      >
        <div className="text-center p-8">
          <div 
            className="w-16 h-16 mx-auto mb-4 rounded-full flex items-center justify-center animate-spin"
            style={{ backgroundColor: '#FF6B35' }}
          >
            <span className="text-white text-2xl">🗺️</span>
          </div>
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Carregando Google Maps</h3>
          <p className="text-sm text-gray-500">Aguarde enquanto o mapa é inicializado...</p>
        </div>
      </div>
    );
  }

  return (
    <div 
      ref={mapRef} 
      className="w-full h-full"
      style={{ minHeight: '400px' }}
    />
  );
};

export default GoogleMap;