// ===== AUTOCOMPLETE COM DROPDOWN =====

// Criar dropdown para autocomplete
function criarDropdownAutocomplete(input, tipo) {
    const formGroup = input.parentElement;
    if (document.getElementById(`dropdown-${tipo}`)) return; // Não duplicar
    
    const dropdown = document.createElement('div');
    dropdown.className = 'autocomplete-dropdown';
    dropdown.id = `dropdown-${tipo}`;
    formGroup.appendChild(dropdown);
}

// Abrir dropdown com resultados
function abrirDropdown(tipo, resultados) {
    const input = tipo === 'origem'
        ? document.getElementById('origem')
        : document.getElementById('destino');

    // Garantir que o dropdown exista (criar se necessário)
    let dropdown = document.getElementById(`dropdown-${tipo}`);
    if (!dropdown) {
        criarDropdownAutocomplete(input, tipo);
        dropdown = document.getElementById(`dropdown-${tipo}`);
    }

    dropdown.innerHTML = '';
    
    resultados.forEach((result) => {
        const item = document.createElement('div');
        item.className = 'autocomplete-item';
        
        const main = document.createElement('div');
        main.className = 'autocomplete-item-main';
        main.textContent = extrairEnderecoPrincipal(result);
        
        const sub = document.createElement('div');
        sub.className = 'autocomplete-item-sub';
        sub.textContent = extrairCidade(result);
        
        item.appendChild(main);
        item.appendChild(sub);
        
        item.addEventListener('click', () => {
            selecionarEndereco(result, tipo, input, dropdown);
        });
        
        dropdown.appendChild(item);
    });

    dropdown.classList.add('show');
}

// Fechar dropdown
function fecharDropdown(tipo) {
    const dropdown = document.getElementById(`dropdown-${tipo}`);
    if (dropdown) {
        dropdown.classList.remove('show');
    }
}

// Extrair apenas a rua e número
function extrairEnderecoPrincipal(result) {
    const parts = result.formatted_address.split(',');
    return parts[0].trim();
}

// Extrair cidade e estado
function extrairCidade(result) {
    const parts = result.formatted_address.split(',');
    if (parts.length >= 2) {
        return parts.slice(-2).join(',').trim();
    }
    return '';
}

// Selecionar endereço do dropdown
function selecionarEndereco(result, tipo, input, dropdown) {
    const coords = {
        lat: result.geometry.location.lat(),
        lng: result.geometry.location.lng()
    };

    input.value = result.formatted_address;

    if (tipo === 'origem') {
        origemCoords = coords;
        console.log('📍 Origem selecionada:', coords);
        if (map) {
            map.setCenter(coords);
        }
    } else {
        destinoCoords = coords;
        console.log('📍 Destino selecionado:', coords);
    }

    // Visual de sucesso
    input.style.borderColor = '#10b981';
    input.style.boxShadow = '0 0 0 3px rgba(16, 185, 129, 0.1)';
    
    dropdown.classList.remove('show');
    verificarEnderecos();
    showNotification(`✅ ${tipo === 'origem' ? 'Origem' : 'Destino'} selecionado`, 'success', 2000);
}

// Buscar endereços com Geocoding
function buscarEnderecosAutocomplete(endereco, tipo) {
    if (!google || !google.maps || !google.maps.Geocoder) {
        console.error('Google Maps Geocoder não disponível');
        return;
    }

    const geocoder = new google.maps.Geocoder();
    const input = tipo === 'origem' 
        ? document.getElementById('origem')
        : document.getElementById('destino');
    
    geocoder.geocode(
        { 
            address: endereco,
            componentRestrictions: { country: 'br' }
        },
        (results, status) => {
            if (status === 'OK' && results.length > 0) {
                console.log(`✅ Encontrados ${results.length} endereços para "${endereco}"`);
                
                // Mostrar nos logs
                results.slice(0, 5).forEach((result, index) => {
                    console.log(`  ${index + 1}. ${extrairEnderecoPrincipal(result)} - ${extrairCidade(result)}`);
                });

                // Abrir dropdown com os primeiros 8 resultados
                abrirDropdown(tipo, results.slice(0, 8));
            } else {
                console.log(`⚠️ Nenhum resultado encontrado para "${endereco}"`);
                fecharDropdown(tipo);
                input.style.borderColor = '#f59e0b';
                showNotification(`Endereço não encontrado`, 'warning', 3000);
            }
        }
    );
}

// Configurar eventos de input com debounce
function configurarEventosAutocomplete() {
    const origemInput = document.getElementById('origem');
    const destinoInput = document.getElementById('destino');

    if (!origemInput || !destinoInput) return;

    // Timers para debounce
    let debounceTimerOrigem = null;
    let debounceTimerDestino = null;

    // Origem
    origemInput.addEventListener('input', function() {
        clearTimeout(debounceTimerOrigem);
        const value = this.value.trim();
        
        // Visual feedback
        if (value.length >= 3) {
            this.style.borderColor = '#3b82f6';
            this.style.boxShadow = '0 0 0 3px rgba(59, 130, 246, 0.1)';
        } else if (value.length > 0) {
            this.style.borderColor = '#f59e0b';
        } else {
            this.style.borderColor = '#e2e8f0';
            this.style.boxShadow = 'none';
            fecharDropdown('origem');
        }
        
        // Debounce: esperar 500ms
        if (value.length >= 3) {
            debounceTimerOrigem = setTimeout(() => {
                console.log('🔍 Buscando origem:', value);
                buscarEnderecosAutocomplete(value, 'origem');
            }, 500);
        }
    });

    // Destino
    destinoInput.addEventListener('input', function() {
        clearTimeout(debounceTimerDestino);
        const value = this.value.trim();
        
        // Visual feedback
        if (value.length >= 3) {
            this.style.borderColor = '#3b82f6';
            this.style.boxShadow = '0 0 0 3px rgba(59, 130, 246, 0.1)';
        } else if (value.length > 0) {
            this.style.borderColor = '#f59e0b';
        } else {
            this.style.borderColor = '#e2e8f0';
            this.style.boxShadow = 'none';
            fecharDropdown('destino');
        }
        
        // Debounce: esperar 500ms
        if (value.length >= 3) {
            debounceTimerDestino = setTimeout(() => {
                console.log('🔍 Buscando destino:', value);
                buscarEnderecosAutocomplete(value, 'destino');
            }, 500);
        }
    });

    // Fechar dropdown ao clicar fora
    document.addEventListener('click', (e) => {
        if (!e.target.closest('.form-group')) {
            fecharDropdown('origem');
            fecharDropdown('destino');
        }
    });
}

// Inicializar quando a página carregar
document.addEventListener('DOMContentLoaded', () => {
    setTimeout(configurarEventosAutocomplete, 1000);
});
