<?php
header('Content-Type: text/html; charset=utf-8');

echo "<h2>🔍 Diagnóstico do Erro de Conexão</h2>";

// 1. Verificar se o motorista existe
echo "<h3>1. Verificando Motorista</h3>";
try {
    require_once 'config/database.php';
    
    $db = new Database();
    $conn = $db->connect();
    
    $stmt = $conn->prepare("SELECT id, nome, email, disponivel FROM motoristas WHERE email = ?");
    $stmt->execute(['andre.rj.tj@gmail.com']);
    $motorista = $stmt->fetch();
    
    if ($motorista) {
        echo "✅ Motorista encontrado:<br>";
        echo "ID: {$motorista['id']}<br>";
        echo "Nome: {$motorista['nome']}<br>";
        echo "Email: {$motorista['email']}<br>";
        echo "Disponível: " . ($motorista['disponivel'] ? 'Sim' : 'Não') . "<br><br>";
    } else {
        echo "❌ Motorista não encontrado<br><br>";
    }
} catch (Exception $e) {
    echo "❌ Erro na consulta: " . $e->getMessage() . "<br><br>";
}

// 2. Testar API diretamente
echo "<h3>2. Testando API de Motoristas</h3>";

// Simular requisição POST
$_SERVER['REQUEST_METHOD'] = 'POST';
$_POST = [];

$test_data = [
    'action' => 'alterar_disponibilidade',
    'motorista_id' => '1',
    'disponivel' => true
];

echo "Dados enviados: <pre>" . json_encode($test_data, JSON_PRETTY_PRINT) . "</pre>";

// Capturar saída
ob_start();

// Simular input JSON
$temp_file = tempnam(sys_get_temp_dir(), 'test_input');
file_put_contents($temp_file, json_encode($test_data));

// Redirecionar stdin temporariamente
$old_stdin = fopen('php://stdin', 'r');
$new_stdin = fopen($temp_file, 'r');

try {
    // Incluir API
    include 'api/motoristas.php';
} catch (Exception $e) {
    echo "❌ Erro na API: " . $e->getMessage() . "<br>";
}

$output = ob_get_clean();

// Restaurar stdin
fclose($new_stdin);
unlink($temp_file);

echo "Resposta da API:<br>";
echo "<pre style='background:#f5f5f5;padding:10px;border-radius:5px;'>$output</pre>";

// 3. Testar conexão direta com banco
echo "<h3>3. Testando Update Direto no Banco</h3>";
try {
    $stmt = $conn->prepare("UPDATE motoristas SET disponivel = ? WHERE id = ?");
    $result = $stmt->execute([1, 1]);
    
    if ($result) {
        echo "✅ Update direto funcionou<br>";
        echo "Linhas afetadas: " . $stmt->rowCount() . "<br>";
        
        // Verificar se realmente atualizou
        $stmt2 = $conn->prepare("SELECT disponivel FROM motoristas WHERE id = ?");
        $stmt2->execute([1]);
        $row = $stmt2->fetch();
        echo "Status atual: " . ($row['disponivel'] ? 'Disponível' : 'Indisponível') . "<br>";
    } else {
        echo "❌ Update direto falhou<br>";
    }
} catch (Exception $e) {
    echo "❌ Erro no update: " . $e->getMessage() . "<br>";
}

// 4. Verificar estrutura da tabela
echo "<h3>4. Estrutura da Tabela Motoristas</h3>";
try {
    $stmt = $conn->query("DESCRIBE motoristas");
    $columns = $stmt->fetchAll();
    
    echo "<table border='1' style='border-collapse:collapse;'>";
    echo "<tr><th>Campo</th><th>Tipo</th><th>Nulo</th><th>Chave</th><th>Padrão</th></tr>";
    foreach ($columns as $col) {
        echo "<tr>";
        echo "<td>{$col['Field']}</td>";
        echo "<td>{$col['Type']}</td>";
        echo "<td>{$col['Null']}</td>";
        echo "<td>{$col['Key']}</td>";
        echo "<td>{$col['Default']}</td>";
        echo "</tr>";
    }
    echo "</table>";
} catch (Exception $e) {
    echo "❌ Erro ao verificar estrutura: " . $e->getMessage() . "<br>";
}

// 5. Verificar logs de erro do PHP
echo "<h3>5. Logs de Erro</h3>";
$error_log = ini_get('error_log');
if ($error_log && file_exists($error_log)) {
    $errors = file_get_contents($error_log);
    $recent_errors = array_slice(explode("\n", $errors), -10);
    echo "<pre style='background:#ffe6e6;padding:10px;border-radius:5px;'>";
    echo implode("\n", $recent_errors);
    echo "</pre>";
} else {
    echo "Nenhum log de erro encontrado ou configurado.<br>";
}

echo "<hr>";
echo "<h3>6. Teste de Conectividade JavaScript</h3>";
echo "<button onclick='testConnection()'>Testar Conexão via JavaScript</button>";
echo "<div id='result'></div>";

echo "<script>
async function testConnection() {
    const result = document.getElementById('result');
    result.innerHTML = '<p>Testando...</p>';
    
    try {
        const response = await fetch('/plataforma/api/motoristas.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'alterar_disponibilidade',
                motorista_id: '1',
                disponivel: true
            })
        });
        
        result.innerHTML = '<h4>Status HTTP: ' + response.status + '</h4>';
        
        if (response.ok) {
            const data = await response.json();
            result.innerHTML += '<pre>' + JSON.stringify(data, null, 2) + '</pre>';
        } else {
            result.innerHTML += '<p style=\"color:red\">Erro HTTP: ' + response.statusText + '</p>';
        }
    } catch (error) {
        result.innerHTML = '<p style=\"color:red\">Erro: ' + error.message + '</p>';
    }
}
</script>";
?>