<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/functions.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

// Solicitar Corrida
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'solicitar_corrida') {
    $usuario_id = $input['usuario_id'] ?? '';
    $categoria_veiculo_id = $input['categoria_veiculo_id'] ?? '';
    $tipo = $input['tipo'] ?? 'passageiro';
    $origem_lat = $input['origem_latitude'] ?? '';
    $origem_lng = $input['origem_longitude'] ?? '';
    $origem_endereco = $input['origem_endereco'] ?? '';
    $destino_lat = $input['destino_latitude'] ?? '';
    $destino_lng = $input['destino_longitude'] ?? '';
    $destino_endereco = $input['destino_endereco'] ?? '';
    $metodo_pagamento = $input['metodo_pagamento'] ?? 'dinheiro';
    
    // Dados de entrega (opcional)
    $descricao_entrega = $input['descricao_entrega'] ?? null;
    $nome_destinatario = $input['nome_destinatario'] ?? null;
    $telefone_destinatario = $input['telefone_destinatario'] ?? null;
    
    if (empty($usuario_id) || empty($categoria_veiculo_id) || empty($origem_lat) || 
        empty($origem_lng) || empty($destino_lat) || empty($destino_lng)) {
        jsonResponse(['success' => false, 'message' => 'Dados incompletos'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Calcular distância e duração estimada
        $distancia_km = calcularDistancia($origem_lat, $origem_lng, $destino_lat, $destino_lng);
        $duracao_minutos = ceil($distancia_km * 3); // Estimativa: 20km/h média
        
        // Obter multiplicador de tarifa dinâmica
        $multiplicador = obterMultiplicadorDinamico($categoria_veiculo_id, $origem_lat, $origem_lng);
        
        // Calcular tarifa
        $tarifa = calcularTarifa($categoria_veiculo_id, $distancia_km, $duracao_minutos, $multiplicador);
        
        if (!$tarifa) {
            jsonResponse(['success' => false, 'message' => 'Erro ao calcular tarifa'], 500);
        }
        
        // Inserir corrida
        $stmt = $conn->prepare("
            INSERT INTO corridas (
                usuario_id, categoria_veiculo_id, tipo,
                origem_latitude, origem_longitude, origem_endereco,
                destino_latitude, destino_longitude, destino_endereco,
                descricao_entrega, nome_destinatario, telefone_destinatario,
                distancia_km, duracao_minutos,
                tarifa_base, tarifa_distancia, tarifa_tempo,
                multiplicador_dinamico, valor_total, taxa_plataforma, valor_motorista,
                metodo_pagamento, status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'solicitada')
        ");
        
        $stmt->execute([
            $usuario_id, $categoria_veiculo_id, $tipo,
            $origem_lat, $origem_lng, $origem_endereco,
            $destino_lat, $destino_lng, $destino_endereco,
            $descricao_entrega, $nome_destinatario, $telefone_destinatario,
            $distancia_km, $duracao_minutos,
            $tarifa['tarifa_base'], $tarifa['tarifa_distancia'], $tarifa['tarifa_tempo'],
            $multiplicador, $tarifa['valor_total'], $tarifa['taxa_plataforma'], $tarifa['valor_motorista'],
            $metodo_pagamento
        ]);
        
        $corrida_id = $conn->lastInsertId();
        
        registrarLog('usuario', $usuario_id, 'solicitar_corrida', "Corrida #$corrida_id solicitada");
        
        jsonResponse([
            'success' => true,
            'message' => 'Corrida solicitada com sucesso',
            'corrida_id' => $corrida_id,
            'tarifa' => $tarifa,
            'multiplicador' => $multiplicador
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao solicitar corrida: ' . $e->getMessage()], 500);
    }
}

// Buscar Motoristas Disponíveis
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'motoristas_disponiveis') {
    $categoria_id = $_GET['categoria_veiculo_id'] ?? '';
    $latitude = $_GET['latitude'] ?? '';
    $longitude = $_GET['longitude'] ?? '';
    $raio_km = $_GET['raio_km'] ?? 10;
    
    if (empty($categoria_id) || empty($latitude) || empty($longitude)) {
        jsonResponse(['success' => false, 'message' => 'Dados incompletos'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            SELECT id, nome, latitude, longitude, avaliacao_media, total_corridas,
                   (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * 
                   cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
                   sin(radians(latitude)))) AS distancia
            FROM motoristas
            WHERE categoria_veiculo_id = ?
            AND status = 'ativo'
            AND disponivel = TRUE
            AND latitude IS NOT NULL
            AND longitude IS NOT NULL
            HAVING distancia <= ?
            ORDER BY distancia ASC
            LIMIT 20
        ");
        
        $stmt->execute([$latitude, $longitude, $latitude, $categoria_id, $raio_km]);
        $motoristas = $stmt->fetchAll();
        
        jsonResponse([
            'success' => true,
            'motoristas' => $motoristas,
            'total' => count($motoristas)
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao buscar motoristas'], 500);
    }
}

// Aceitar Corrida (Motorista)
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'aceitar_corrida') {
    $corrida_id = $input['corrida_id'] ?? '';
    $motorista_id = $input['motorista_id'] ?? '';
    
    if (empty($corrida_id) || empty($motorista_id)) {
        jsonResponse(['success' => false, 'message' => 'Dados incompletos'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $conn->beginTransaction();
        
        // Verificar se corrida ainda está disponível
        $stmt = $conn->prepare("SELECT status, usuario_id FROM corridas WHERE id = ? FOR UPDATE");
        $stmt->execute([$corrida_id]);
        $corrida = $stmt->fetch();
        
        if (!$corrida || $corrida['status'] !== 'solicitada') {
            $conn->rollBack();
            jsonResponse(['success' => false, 'message' => 'Corrida não disponível'], 400);
        }
        
        // Atualizar corrida
        $stmt = $conn->prepare("
            UPDATE corridas 
            SET motorista_id = ?, status = 'aceita', aceita_em = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$motorista_id, $corrida_id]);
        
        // Atualizar motorista
        $stmt = $conn->prepare("UPDATE motoristas SET disponivel = FALSE WHERE id = ?");
        $stmt->execute([$motorista_id]);
        
        $conn->commit();
        
        registrarLog('motorista', $motorista_id, 'aceitar_corrida', "Corrida #$corrida_id aceita");
        enviarNotificacao('usuario', $corrida['usuario_id'], 'Motorista a caminho!', 'Seu motorista aceitou a corrida', 'corrida');
        
        jsonResponse([
            'success' => true,
            'message' => 'Corrida aceita com sucesso'
        ]);
        
    } catch (Exception $e) {
        $conn->rollBack();
        jsonResponse(['success' => false, 'message' => 'Erro ao aceitar corrida'], 500);
    }
}

// Iniciar Corrida
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'iniciar_corrida') {
    $corrida_id = $input['corrida_id'] ?? '';
    $motorista_id = $input['motorista_id'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            UPDATE corridas 
            SET status = 'em_andamento', iniciada_em = NOW()
            WHERE id = ? AND motorista_id = ? AND status = 'aceita'
        ");
        $stmt->execute([$corrida_id, $motorista_id]);
        
        if ($stmt->rowCount() === 0) {
            jsonResponse(['success' => false, 'message' => 'Não foi possível iniciar a corrida'], 400);
        }
        
        registrarLog('motorista', $motorista_id, 'iniciar_corrida', "Corrida #$corrida_id iniciada");
        
        jsonResponse(['success' => true, 'message' => 'Corrida iniciada']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao iniciar corrida'], 500);
    }
}

// Concluir Corrida
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'concluir_corrida') {
    $corrida_id = $input['corrida_id'] ?? '';
    $motorista_id = $input['motorista_id'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $conn->beginTransaction();
        
        // Buscar dados da corrida
        $stmt = $conn->prepare("
            SELECT valor_motorista, usuario_id 
            FROM corridas 
            WHERE id = ? AND motorista_id = ? AND status = 'em_andamento'
        ");
        $stmt->execute([$corrida_id, $motorista_id]);
        $corrida = $stmt->fetch();
        
        if (!$corrida) {
            $conn->rollBack();
            jsonResponse(['success' => false, 'message' => 'Corrida não encontrada'], 400);
        }
        
        // Atualizar corrida
        $stmt = $conn->prepare("
            UPDATE corridas 
            SET status = 'concluida', concluida_em = NOW(), status_pagamento = 'pago'
            WHERE id = ?
        ");
        $stmt->execute([$corrida_id]);
        
        // Atualizar saldo do motorista
        $stmt = $conn->prepare("
            UPDATE motoristas 
            SET saldo = saldo + ?, total_corridas = total_corridas + 1, disponivel = TRUE
            WHERE id = ?
        ");
        $stmt->execute([$corrida['valor_motorista'], $motorista_id]);
        
        // Registrar transação
        $stmt = $conn->prepare("
            INSERT INTO transacoes (corrida_id, motorista_id, tipo, valor, descricao, status)
            VALUES (?, ?, 'corrida', ?, 'Pagamento de corrida', 'concluida')
        ");
        $stmt->execute([$corrida_id, $motorista_id, $corrida['valor_motorista']]);
        
        $conn->commit();
        
        registrarLog('motorista', $motorista_id, 'concluir_corrida', "Corrida #$corrida_id concluída");
        enviarNotificacao('usuario', $corrida['usuario_id'], 'Corrida concluída!', 'Avalie sua experiência', 'corrida');
        
        jsonResponse(['success' => true, 'message' => 'Corrida concluída com sucesso']);
        
    } catch (Exception $e) {
        $conn->rollBack();
        jsonResponse(['success' => false, 'message' => 'Erro ao concluir corrida'], 500);
    }
}

// Cancelar Corrida
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'cancelar_corrida') {
    $corrida_id = $input['corrida_id'] ?? '';
    $cancelado_por = $input['cancelado_por'] ?? '';
    $usuario_id = $input['usuario_id'] ?? '';
    $motivo = $input['motivo'] ?? '';
    
    if (empty($corrida_id) || empty($cancelado_por) || empty($usuario_id)) {
        jsonResponse(['success' => false, 'message' => 'Dados incompletos para cancelamento'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Primeiro, verificar se a corrida existe e seu status atual
        $stmt = $conn->prepare("SELECT id, status, usuario_id FROM corridas WHERE id = ?");
        $stmt->execute([$corrida_id]);
        $corrida = $stmt->fetch();
        
        if (!$corrida) {
            jsonResponse(['success' => false, 'message' => 'Corrida não encontrada'], 404);
        }
        
        if ($corrida['usuario_id'] != $usuario_id) {
            jsonResponse(['success' => false, 'message' => 'Você não tem permissão para cancelar esta corrida'], 403);
        }
        
        if (!in_array($corrida['status'], ['solicitada', 'aceita'])) {
            jsonResponse(['success' => false, 'message' => "Corrida não pode ser cancelada. Status atual: {$corrida['status']}"], 400);
        }
        
        // Agora cancelar a corrida
        $stmt = $conn->prepare("
            UPDATE corridas 
            SET status = 'cancelada', cancelado_por = ?, motivo_cancelamento = ?, cancelada_em = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$cancelado_por, $motivo, $corrida_id]);
        
        if ($stmt->rowCount() === 0) {
            jsonResponse(['success' => false, 'message' => 'Falha ao atualizar status da corrida'], 500);
        }
        
        // Liberar motorista se já estava aceita
        $stmt = $conn->prepare("
            UPDATE motoristas m
            INNER JOIN corridas c ON c.motorista_id = m.id
            SET m.disponivel = TRUE
            WHERE c.id = ?
        ");
        $stmt->execute([$corrida_id]);
        
        registrarLog($cancelado_por, $usuario_id, 'cancelar_corrida', "Corrida #$corrida_id cancelada");
        
        jsonResponse(['success' => true, 'message' => 'Corrida cancelada']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao cancelar corrida'], 500);
    }
}

// Avaliar Corrida
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'avaliar_corrida') {
    $corrida_id = $input['corrida_id'] ?? '';
    $avaliado_por = $input['avaliado_por'] ?? 'usuario';
    $avaliacao = $input['avaliacao'] ?? 0;
    $comentario = $input['comentario'] ?? '';
    
    if ($avaliacao < 1 || $avaliacao > 5) {
        jsonResponse(['success' => false, 'message' => 'Avaliação deve ser entre 1 e 5'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        if ($avaliado_por === 'usuario') {
            $stmt = $conn->prepare("
                UPDATE corridas 
                SET avaliacao_motorista = ?, comentario_motorista = ?
                WHERE id = ? AND status = 'concluida'
            ");
            $stmt->execute([$avaliacao, $comentario, $corrida_id]);
            
            // Atualizar média do motorista
            $stmt = $conn->prepare("
                UPDATE motoristas m
                INNER JOIN corridas c ON c.motorista_id = m.id
                SET m.avaliacao_media = (
                    SELECT AVG(avaliacao_motorista) 
                    FROM corridas 
                    WHERE motorista_id = m.id AND avaliacao_motorista IS NOT NULL
                ),
                m.total_avaliacoes = m.total_avaliacoes + 1
                WHERE c.id = ?
            ");
            $stmt->execute([$corrida_id]);
        } else {
            $stmt = $conn->prepare("
                UPDATE corridas 
                SET avaliacao_usuario = ?, comentario_usuario = ?
                WHERE id = ? AND status = 'concluida'
            ");
            $stmt->execute([$avaliacao, $comentario, $corrida_id]);
            
            // Atualizar média do usuário
            $stmt = $conn->prepare("
                UPDATE usuarios u
                INNER JOIN corridas c ON c.usuario_id = u.id
                SET u.avaliacao_media = (
                    SELECT AVG(avaliacao_usuario) 
                    FROM corridas 
                    WHERE usuario_id = u.id AND avaliacao_usuario IS NOT NULL
                ),
                u.total_avaliacoes = u.total_avaliacoes + 1
                WHERE c.id = ?
            ");
            $stmt->execute([$corrida_id]);
        }
        
        jsonResponse(['success' => true, 'message' => 'Avaliação registrada com sucesso']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao avaliar corrida'], 500);
    }
}

// Listar Corridas
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_corridas') {
    $tipo_usuario = $_GET['tipo_usuario'] ?? '';
    $usuario_id = $_GET['usuario_id'] ?? '';
    $status = $_GET['status'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $sql = "SELECT c.*, u.nome as usuario_nome, m.nome as motorista_nome, 
                       cv.nome as categoria_nome
                FROM corridas c
                LEFT JOIN usuarios u ON c.usuario_id = u.id
                LEFT JOIN motoristas m ON c.motorista_id = m.id
                LEFT JOIN categorias_veiculo cv ON c.categoria_veiculo_id = cv.id
                WHERE 1=1";
        
        $params = [];
        
        if ($tipo_usuario === 'usuario') {
            $sql .= " AND c.usuario_id = ?";
            $params[] = $usuario_id;
        } elseif ($tipo_usuario === 'motorista') {
            $sql .= " AND c.motorista_id = ?";
            $params[] = $usuario_id;
        }
        
        if (!empty($status)) {
            $sql .= " AND c.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY c.solicitada_em DESC LIMIT 50";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $corridas = $stmt->fetchAll();
        
        jsonResponse([
            'success' => true,
            'corridas' => $corridas
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar corridas'], 500);
    }
}

// Buscar Corridas Disponíveis para Motorista
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'buscar_corridas_disponiveis') {
    $motorista_id = $_GET['motorista_id'] ?? '';
    
    if (empty($motorista_id)) {
        jsonResponse(['success' => false, 'message' => 'ID do motorista é obrigatório'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Buscar categoria do motorista
        $stmt = $conn->prepare("SELECT categoria_veiculo_id, latitude, longitude FROM motoristas WHERE id = ?");
        $stmt->execute([$motorista_id]);
        $motorista = $stmt->fetch();
        
        if (!$motorista) {
            jsonResponse(['success' => false, 'message' => 'Motorista não encontrado'], 404);
        }
        
        // Buscar corridas disponíveis para a categoria do motorista
        $stmt = $conn->prepare("
            SELECT c.*, u.nome as usuario_nome, u.telefone as usuario_telefone
            FROM corridas c
            INNER JOIN usuarios u ON c.usuario_id = u.id
            WHERE c.categoria_veiculo_id = ?
            AND c.status = 'solicitada'
            AND c.motorista_id IS NULL
            ORDER BY c.solicitada_em ASC
            LIMIT 5
        ");
        $stmt->execute([$motorista['categoria_veiculo_id']]);
        $corridas = $stmt->fetchAll();
        
        jsonResponse([
            'success' => true,
            'corridas' => $corridas
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao buscar corridas'], 500);
    }
}

// Obter Corrida Específica
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'obter_corrida') {
    $corrida_id = $_GET['corrida_id'] ?? '';
    
    if (empty($corrida_id)) {
        jsonResponse(['success' => false, 'message' => 'ID da corrida é obrigatório'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            SELECT c.*, u.nome as usuario_nome, u.telefone as usuario_telefone,
                   m.nome as motorista_nome
            FROM corridas c
            LEFT JOIN usuarios u ON c.usuario_id = u.id
            LEFT JOIN motoristas m ON c.motorista_id = m.id
            WHERE c.id = ?
        ");
        $stmt->execute([$corrida_id]);
        $corrida = $stmt->fetch();
        
        if (!$corrida) {
            jsonResponse(['success' => false, 'message' => 'Corrida não encontrada'], 404);
        }
        
        jsonResponse([
            'success' => true,
            'corrida' => $corrida
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao obter corrida'], 500);
    }
}

// Listar corridas do motorista com ganhos
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_corridas_motorista') {
    $motorista_id = $_GET['motorista_id'] ?? '';
    
    if (empty($motorista_id)) {
        jsonResponse(['success' => false, 'message' => 'ID do motorista obrigatório'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Ganhos totais
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as corridas_total,
                SUM(valor_motorista) as ganhos_total,
                AVG(valor_motorista) as ticket_medio
            FROM corridas
            WHERE motorista_id = ? AND status = 'concluida'
        ");
        $stmt->execute([$motorista_id]);
        $totais = $stmt->fetch();
        
        // Ganhos de hoje
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as corridas_hoje,
                SUM(valor_motorista) as ganhos_hoje
            FROM corridas
            WHERE motorista_id = ? AND status = 'concluida' AND DATE(concluida_em) = CURDATE()
        ");
        $stmt->execute([$motorista_id]);
        $hoje = $stmt->fetch();
        
        jsonResponse([
            'success' => true,
            'corridas_total' => $totais['corridas_total'] ?? 0,
            'ganhos_total' => number_format($totais['ganhos_total'] ?? 0, 2, '.', ''),
            'ticket_medio' => number_format($totais['ticket_medio'] ?? 0, 2, '.', ''),
            'corridas_hoje' => $hoje['corridas_hoje'] ?? 0,
            'ganhos_hoje' => number_format($hoje['ganhos_hoje'] ?? 0, 2, '.', '')
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar ganhos: ' . $e->getMessage()], 500);
    }
}

