<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/functions.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

// Listar Usuários
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_usuarios') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $status = $_GET['status'] ?? '';
        $sql = "SELECT id, nome, email, telefone, cpf, status, avaliacao_media, 
                       total_avaliacoes, criado_em 
                FROM usuarios";
        
        if (!empty($status)) {
            $sql .= " WHERE status = ?";
            $stmt = $conn->prepare($sql);
            $stmt->execute([$status]);
        } else {
            $stmt = $conn->query($sql);
        }
        
        $usuarios = $stmt->fetchAll();
        
        jsonResponse(['success' => true, 'usuarios' => $usuarios]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar usuários'], 500);
    }
}

// Listar Motoristas
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_motoristas') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $status = $_GET['status'] ?? '';
        $sql = "SELECT m.*, cv.nome as categoria_nome
                FROM motoristas m
                LEFT JOIN categorias_veiculo cv ON m.categoria_veiculo_id = cv.id";
        
        if (!empty($status)) {
            $sql .= " WHERE m.status = ?";
            $stmt = $conn->prepare($sql);
            $stmt->execute([$status]);
        } else {
            $stmt = $conn->query($sql);
        }
        
        $motoristas = $stmt->fetchAll();
        
        jsonResponse(['success' => true, 'motoristas' => $motoristas]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar motoristas'], 500);
    }
}

// Bloquear/Desbloquear Usuário
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'alterar_status_usuario') {
    $usuario_id = $input['usuario_id'] ?? '';
    $status = $input['status'] ?? '';
    $admin_id = $input['admin_id'] ?? '';
    
    if (!in_array($status, ['ativo', 'bloqueado', 'inativo'])) {
        jsonResponse(['success' => false, 'message' => 'Status inválido'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("UPDATE usuarios SET status = ? WHERE id = ?");
        $stmt->execute([$status, $usuario_id]);
        
        registrarLog('admin', $admin_id, 'alterar_status_usuario', "Usuário #$usuario_id alterado para $status");
        enviarNotificacao('usuario', $usuario_id, 'Status da conta alterado', "Sua conta foi alterada para: $status", 'alerta');
        
        jsonResponse(['success' => true, 'message' => 'Status atualizado com sucesso']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao atualizar status'], 500);
    }
}

// Bloquear/Desbloquear/Aprovar Motorista
if ($method === 'POST' && isset($input['action']) && $input['action'] === 'alterar_status_motorista') {
    $motorista_id = $input['motorista_id'] ?? '';
    $status = $input['status'] ?? '';
    $admin_id = $input['admin_id'] ?? '';
    
    if (!in_array($status, ['ativo', 'bloqueado', 'inativo', 'pendente'])) {
        jsonResponse(['success' => false, 'message' => 'Status inválido'], 400);
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("UPDATE motoristas SET status = ? WHERE id = ?");
        $stmt->execute([$status, $motorista_id]);
        
        $mensagem = $status === 'ativo' ? 'Seu cadastro foi aprovado! Você já pode começar a trabalhar.' : 
                    "Status da sua conta alterado para: $status";
        
        registrarLog('admin', $admin_id, 'alterar_status_motorista', "Motorista #$motorista_id alterado para $status");
        enviarNotificacao('motorista', $motorista_id, 'Status da conta alterado', $mensagem, 'alerta');
        
        jsonResponse(['success' => true, 'message' => 'Status atualizado com sucesso']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao atualizar status'], 500);
    }
}

// Dashboard - Estatísticas
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'dashboard') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Total de usuários
        $stmt = $conn->query("SELECT COUNT(*) as total, 
                              SUM(CASE WHEN status = 'ativo' THEN 1 ELSE 0 END) as ativos,
                              SUM(CASE WHEN status = 'bloqueado' THEN 1 ELSE 0 END) as bloqueados
                              FROM usuarios");
        $usuarios = $stmt->fetch();
        
        // Total de motoristas
        $stmt = $conn->query("SELECT COUNT(*) as total,
                              SUM(CASE WHEN status = 'ativo' THEN 1 ELSE 0 END) as ativos,
                              SUM(CASE WHEN status = 'bloqueado' THEN 1 ELSE 0 END) as bloqueados,
                              SUM(CASE WHEN status = 'pendente' THEN 1 ELSE 0 END) as pendentes,
                              SUM(CASE WHEN disponivel = TRUE THEN 1 ELSE 0 END) as disponiveis
                              FROM motoristas");
        $motoristas = $stmt->fetch();
        
        // Corridas
        $stmt = $conn->query("SELECT COUNT(*) as total,
                              SUM(CASE WHEN status = 'concluida' THEN 1 ELSE 0 END) as concluidas,
                              SUM(CASE WHEN status = 'em_andamento' THEN 1 ELSE 0 END) as em_andamento,
                              SUM(CASE WHEN status = 'cancelada' THEN 1 ELSE 0 END) as canceladas,
                              SUM(CASE WHEN status = 'concluida' THEN valor_total ELSE 0 END) as receita_total,
                              SUM(CASE WHEN status = 'concluida' THEN taxa_plataforma ELSE 0 END) as taxa_plataforma_total
                              FROM corridas");
        $corridas = $stmt->fetch();
        
        // Corridas hoje
        $stmt = $conn->query("SELECT COUNT(*) as total,
                              SUM(valor_total) as receita
                              FROM corridas 
                              WHERE DATE(solicitada_em) = CURDATE() AND status = 'concluida'");
        $hoje = $stmt->fetch();
        
        // Corridas por categoria
        $stmt = $conn->query("SELECT cv.nome, COUNT(*) as total, SUM(c.valor_total) as receita
                              FROM corridas c
                              INNER JOIN categorias_veiculo cv ON c.categoria_veiculo_id = cv.id
                              WHERE c.status = 'concluida'
                              GROUP BY cv.id
                              ORDER BY total DESC");
        $por_categoria = $stmt->fetchAll();
        
        jsonResponse([
            'success' => true,
            'usuarios' => $usuarios,
            'motoristas' => $motoristas,
            'corridas' => $corridas,
            'hoje' => $hoje,
            'por_categoria' => $por_categoria
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao carregar dashboard'], 500);
    }
}

// Listar Todas as Corridas (Admin)
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_todas_corridas') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $limit = $_GET['limit'] ?? 100;
        $offset = $_GET['offset'] ?? 0;
        
        $stmt = $conn->prepare("
            SELECT c.*, u.nome as usuario_nome, u.email as usuario_email,
                   m.nome as motorista_nome, m.email as motorista_email,
                   cv.nome as categoria_nome
            FROM corridas c
            LEFT JOIN usuarios u ON c.usuario_id = u.id
            LEFT JOIN motoristas m ON c.motorista_id = m.id
            LEFT JOIN categorias_veiculo cv ON c.categoria_veiculo_id = cv.id
            ORDER BY c.solicitada_em DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$limit, $offset]);
        $corridas = $stmt->fetchAll();
        
        jsonResponse(['success' => true, 'corridas' => $corridas]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar corridas'], 500);
    }
}

// Gerenciar Categorias
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_categorias') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->query("SELECT * FROM categorias_veiculo ORDER BY tipo, nome");
        $categorias = $stmt->fetchAll();
        
        jsonResponse(['success' => true, 'categorias' => $categorias]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar categorias'], 500);
    }
}

if ($method === 'POST' && isset($input['action']) && $input['action'] === 'atualizar_categoria') {
    $categoria_id = $input['categoria_id'] ?? '';
    $tarifa_base = $input['tarifa_base'] ?? '';
    $tarifa_por_km = $input['tarifa_por_km'] ?? '';
    $tarifa_por_minuto = $input['tarifa_por_minuto'] ?? '';
    $ativo = $input['ativo'] ?? true;
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            UPDATE categorias_veiculo 
            SET tarifa_base = ?, tarifa_por_km = ?, tarifa_por_minuto = ?, ativo = ?
            WHERE id = ?
        ");
        $stmt->execute([$tarifa_base, $tarifa_por_km, $tarifa_por_minuto, $ativo ? 1 : 0, $categoria_id]);
        
        jsonResponse(['success' => true, 'message' => 'Categoria atualizada com sucesso']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao atualizar categoria'], 500);
    }
}

// Gerenciar Configurações
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'listar_configuracoes') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->query("SELECT * FROM configuracoes ORDER BY chave");
        $configuracoes = $stmt->fetchAll();
        
        jsonResponse(['success' => true, 'configuracoes' => $configuracoes]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao listar configurações'], 500);
    }
}

if ($method === 'POST' && isset($input['action']) && $input['action'] === 'atualizar_configuracao') {
    $chave = $input['chave'] ?? '';
    $valor = $input['valor'] ?? '';
    $admin_id = $input['admin_id'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("UPDATE configuracoes SET valor = ? WHERE chave = ?");
        $stmt->execute([$valor, $chave]);
        
        registrarLog('admin', $admin_id, 'atualizar_configuracao', "Configuração $chave atualizada");
        
        jsonResponse(['success' => true, 'message' => 'Configuração atualizada com sucesso']);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao atualizar configuração'], 500);
    }
}

// Relatórios Financeiros
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'relatorio_financeiro') {
    $data_inicio = $_GET['data_inicio'] ?? date('Y-m-01');
    $data_fim = $_GET['data_fim'] ?? date('Y-m-d');
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            SELECT 
                DATE(concluida_em) as data,
                COUNT(*) as total_corridas,
                SUM(valor_total) as receita_total,
                SUM(taxa_plataforma) as taxa_plataforma,
                SUM(valor_motorista) as valor_motoristas,
                AVG(valor_total) as ticket_medio
            FROM corridas
            WHERE status = 'concluida'
            AND DATE(concluida_em) BETWEEN ? AND ?
            GROUP BY DATE(concluida_em)
            ORDER BY data DESC
        ");
        $stmt->execute([$data_inicio, $data_fim]);
        $relatorio = $stmt->fetchAll();
        
        // Totais
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as total_corridas,
                SUM(valor_total) as receita_total,
                SUM(taxa_plataforma) as taxa_plataforma,
                SUM(valor_motorista) as valor_motoristas
            FROM corridas
            WHERE status = 'concluida'
            AND DATE(concluida_em) BETWEEN ? AND ?
        ");
        $stmt->execute([$data_inicio, $data_fim]);
        $totais = $stmt->fetch();
        
        jsonResponse([
            'success' => true,
            'relatorio' => $relatorio,
            'totais' => $totais
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao gerar relatório'], 500);
    }
}

// Mapa em Tempo Real
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'mapa_tempo_real') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Motoristas online
        $stmt = $conn->query("
            SELECT id, nome, latitude, longitude, disponivel, categoria_veiculo_id
            FROM motoristas
            WHERE status = 'ativo'
            AND latitude IS NOT NULL
            AND longitude IS NOT NULL
            AND ultima_localizacao > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
        ");
        $motoristas = $stmt->fetchAll();
        
        // Corridas em andamento
        $stmt = $conn->query("
            SELECT c.id, c.origem_latitude, c.origem_longitude, 
                   c.destino_latitude, c.destino_longitude, c.status,
                   u.nome as usuario_nome, m.nome as motorista_nome
            FROM corridas c
            LEFT JOIN usuarios u ON c.usuario_id = u.id
            LEFT JOIN motoristas m ON c.motorista_id = m.id
            WHERE c.status IN ('solicitada', 'aceita', 'em_andamento')
        ");
        $corridas = $stmt->fetchAll();
        
        jsonResponse([
            'success' => true,
            'motoristas' => $motoristas,
            'corridas' => $corridas
        ]);
        
    } catch (Exception $e) {
        jsonResponse(['success' => false, 'message' => 'Erro ao carregar mapa'], 500);
    }
}

jsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
