# 🔧 Instruções para Debug da Conexão do Motorista

## Problema Identificado
O usuário está enfrentando "Erro de conexão" ao clicar no botão "Conectar" no app PHP do motorista.

## Arquivos Criados para Debug

### 1. `motorista_debug.html`
Página de diagnóstico completa que permite:
- Testar login do motorista
- Testar conexão com a API
- Testar múltiplos caminhos da API
- Ver logs do console em tempo real

**Como usar:**
1. Abra `http://localhost/plataforma/motorista_debug.html` no navegador
2. Teste o login com as credenciais: `andre.rj.tj@gmail.com` / `123456`
3. Após o login, teste a conexão
4. Se falhar, use "Testar Todos os Caminhos" para encontrar o caminho correto

### 2. `test_connection_simple.php`
API de teste simplificada que registra logs detalhados.

### 3. `test_connection_js.html`
Página para testar a API via JavaScript.

## Melhorias Implementadas no `motorista/index.html`

### 1. Detecção Automática do Caminho da API
```javascript
const currentPath = window.location.pathname;
const basePath = currentPath.includes('/plataforma/') ? '/plataforma' : '';
const API_URL = basePath + '/api';
```

### 2. Melhor Tratamento de Erros
- Logs detalhados no console
- Tentativa com caminhos alternativos da API
- Mensagens de erro mais específicas
- Modo offline como fallback

### 3. Função de Teste Global
Adicionada função `window.testAPI()` que pode ser chamada no console do navegador para testar diferentes URLs da API.

## Como Diagnosticar o Problema

### Passo 1: Verificar Estrutura de Arquivos
Confirme que os arquivos estão na estrutura correta:
```
/plataforma/
├── api/
│   ├── motoristas.php
│   └── auth.php
├── motorista/
│   └── index.html
└── config/
    └── database.php
```

### Passo 2: Testar no Console do Navegador
1. Abra `motorista/index.html`
2. Faça login com `andre.rj.tj@gmail.com` / `123456`
3. Abra o console do navegador (F12)
4. Execute: `testAPI()`
5. Veja qual URL funciona

### Passo 3: Verificar Logs do Servidor
Se usando Apache/Nginx, verifique os logs de erro para identificar problemas no servidor.

### Passo 4: Testar Manualmente
Use a página `motorista_debug.html` para fazer testes detalhados.

## Possíveis Causas do Problema

### 1. Caminho Incorreto da API
- A URL `/plataforma/api/motoristas.php` pode não existir
- O servidor pode estar configurado diferentemente

### 2. Problemas de CORS
- Headers de CORS podem estar bloqueando a requisição

### 3. Erro no Servidor PHP
- Erro de sintaxe no `api/motoristas.php`
- Problema de conexão com o banco de dados

### 4. Configuração do Servidor Web
- Rewrite rules podem estar interferindo
- Permissões de arquivo incorretas

## Soluções Implementadas

### 1. Detecção Automática de Caminho
O app agora detecta automaticamente o caminho correto da API.

### 2. Fallback para Modo Offline
Se a API não funcionar, o app entra em modo offline com corridas simuladas.

### 3. Tentativas com Múltiplos Caminhos
O app tenta diferentes URLs da API automaticamente.

### 4. Logs Detalhados
Todos os erros são logados no console com detalhes específicos.

## Próximos Passos

1. **Execute o debug**: Use `motorista_debug.html` para identificar o problema exato
2. **Verifique os logs**: Olhe o console do navegador para mensagens de erro
3. **Teste caminhos alternativos**: Use a função `testAPI()` no console
4. **Verifique o servidor**: Confirme que o servidor web está funcionando corretamente

## Comandos Úteis para Debug

### No Console do Navegador:
```javascript
// Testar API
testAPI()

// Ver motorista atual
console.log('currentDriver:', currentDriver)

// Ver URL da API
console.log('API_URL:', API_URL)

// Testar conexão manual
fetch('./api/motoristas.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
        action: 'alterar_disponibilidade',
        motorista_id: '1',
        disponivel: true
    })
}).then(r => r.json()).then(console.log)
```

## Contato
Se o problema persistir após seguir estas instruções, forneça:
1. Screenshot do console do navegador
2. Resultado do `motorista_debug.html`
3. Estrutura de arquivos do seu projeto
4. Configuração do servidor web (Apache/Nginx)